<?php

namespace App\Admin\Controllers;

use OpenAdmin\Admin\Controllers\AdminController;
use OpenAdmin\Admin\Form;
use OpenAdmin\Admin\Grid;
use OpenAdmin\Admin\Show;
use \App\Models\Page;
use \App\Models\SectionTemplate;

class PageController extends AdminController
{
    /**
     * Title for current resource.
     *
     * @var string
     */
    protected $title = 'Page';

    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        $grid = new Grid(new Page());

        $grid->filter(function($filter){
            // Remove the default id filter
            $filter->disableIdFilter();
            $filter->like('name', 'Page Name');
        });

        $grid->column('id', __('Id'));
        $grid->column('name', __('Name'));
        $grid->column('slug', __('link'));
        $grid->column('status', __('Status'))
            ->using([0 => 'Inactive', 1 => 'Active'])
            ->label([0 => 'danger', 1 => 'success']);
        $grid->column('created_at', __('Created at'))->display(function ($value) {
            return \Carbon\Carbon::parse($value)->format('d M Y, H:i');
        });
        $grid->column('updated_at', __('Updated at'))->display(function ($value) {
            return \Carbon\Carbon::parse($value)->format('d M Y, H:i');
        });

        $grid->actions(function ($actions) {
            $actions->disableDelete(); // Removes the delete button
            $actions->disableEdit();   // Removes the edit button
            // $actions->disableView();   // Removes the view button
        });

        // $grid->disableCreateButton();

        return $grid;
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     * @return Show
     */
    protected function detail($id)
    {
        $show = new Show(Page::findOrFail($id));

        $show->field('id', __('Id'));
        $show->field('name', __('Name'));
        $show->field('slug', __('link'));
        $show->field('status', __('Status'))->as(function ($status) {
            return $status ? 'Active' : 'Inactive';
        });

        $show->sections('Section', function ($section) {
            $section->resource('/admin/page-sections');
            $section->column('hint_image', __('Hint Image'))->display(function () {
                return "<img src='" . asset('section_hint_image/' . $this->section_name.'.png') . "' style='max-width:200px;max-height:100px' />";
            });
            $section->column('section_name', __('Section Name'));
            $section->column('created_at', __('Created at'))->display(function ($value) {
                return \Carbon\Carbon::parse($value)->format('d M Y, H:i');
            });

            $section->actions(function ($actions) {
                $actions->disableView();
            });
        });
        
        $show->field('created_at', __('Created at'))->as(function ($value) {
            return \Carbon\Carbon::parse($value)->format('d M Y, H:i');
        });
        $show->field('updated_at', __('Updated at'))->as(function ($value) {
            return \Carbon\Carbon::parse($value)->format('d M Y, H:i');
        });

        return $show;
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */

    protected function form()
    {
        $form = new Form(new Page());

        // Disable the delete button on the edit form
        $form->tools(function (Form\Tools $tools) {
            $tools->disableDelete(); // Hides the delete button from the tools section
            $tools->disableView();   // Hides the view button from the tools section
            // $tools->disableList();  // Hides the list button
        });

        if($form->isCreating()) {
            $form->text('name', __('Name'))
            ->rules('nullable|unique:pages,name')
            ->required();
            $form->switch('status', __('Status'))->default(1);
            $form->hasMany('sections', 'Sections', function (Form\NestedForm $form) {
                $form->select('section_name', 'Type')
                    ->options([
                        'home_services_section' => 'Home Services Section',
                        'home_hero_section' => 'Home Hero Section',
                        'home_choosing_topfilings_section' => 'Home Choosing Topfilings Section',
                        'home_counter_section' => 'Home Counter Section',
                        'home_verified_section' => 'Home Verified Section',
                        'home_blog_section' => 'Blog Section',
                        'home_testimonial_section' => 'Home Testimonial Section',
                        'service_content_form_section' => 'Service Content + Form Section',
                        'service_why_section' => 'Service Why Section',
                        'service_icon_box' => 'Service Icon Box Section',
                        'service_business_structure' => 'Service Business Structure Chart Section',
                        'parallax_section' => 'Parallax Section',
                        'two_column_list' => 'Two Column List',
                        'image_content_section' => 'Image Content Section',
                        'virtual_cfo_why_section' => 'Virtual CFO Why Section',
                        'content' => 'Content Section',
                        'title_paragraph_card' => 'Title Paragraph Card',
                    ])
                    ->required();
            });
        }

        $form->editing(function (Form $form) {
            $pageJson = $form->model()->page_json;
            // about-us page --------------------------------------------------------------------
            if ($form->model()->slug == 'about-us') {
                $form->embeds('page_json', 'About Us', function ($json) use($pageJson) {
                    $json->text('sub_heading', 'Sub Heading')->required();
                    $json->text('heading', 'Section Heading')->required();
                    $json->textarea('paragraph', 'Paragraph')->required();
                    $json->number('year_count', 'Year Experence Count')->required();
                    //Founder---------------------------------------------------------------------------
                    $json->divider('Founder');
                    $json->image('founder_image', 'Founder Image')
                        ->uniqueName()
                        ->move('sections')
                        ->help('Upload image for the section')
                        ->value($pageJson['founder_image'] ?? null);
                    if(isset($pageJson['founder_image'])) {
                        $json->hidden('current_founder_image', 'Current Image')
                            ->default($pageJson['founder_image'])
                            ->readonly();
                    }
                    $json->ckeditor('founder_details', 'Founder Details');
                    //Image-Box Cards -------------------------------------------------------------------
                    $json->divider('Image-Box Cards');
                    for($i=0; $i<4; $i++){
                        $json->embeds('card_'.$i, 'Card '.$i+1, function ($json) use($pageJson, $i) {
                            $json->image('image', 'Image')
                                ->uniqueName()
                                ->move('sections')
                                ->help('Upload image for the section')
                                ->value($pageJson['card_'.$i]['image'] ?? null);
                            if(isset($pageJson['card_'.$i]['image'])) {
                                $json->hidden('current_image', 'Current Image')
                                    ->default($pageJson['card_'.$i]['image'])
                                    ->readonly();
                            }
                            $json->text('title', 'Title');
                            $json->text('paragraph', 'Paragraph');
                        });
                    }
                    //Image-title section-----------------------------------------------------------------
                    $json->divider('Image-Title Section');
                    for($i=0; $i<3; $i++){
                        $json->embeds('section_'.$i, 'Section '.$i+1, function ($json) use($pageJson, $i){
                            $json->image('image', 'Image')
                                ->uniqueName()
                                ->move('sections')
                                ->help('Upload image for the section')
                                ->value($pageJson['section_'.$i]['image'] ?? null);
                            if(isset($pageJson['section_'.$i]['image'])) {
                                $json->hidden('current_image', 'Current Image')
                                    ->default($pageJson['section_'.$i]['image'])
                                    ->readonly();
                            }
                            $json->text('title', 'Title');
                        });
                    }
                    //Counters --------------------------------------------------------------------------
                    $json->divider('Counter Section');
                    $json->number('customer_served', 'Customer Served')->required();
                    $json->number('successfull_starts', 'Successfull Starts')->required();
                    $json->number('satisfaction_rate', 'Satisfaction rate')->required();

                    //Our Mission -----------------------------------------------------------------
                    $json->divider('Our Mission');
                    $json->embeds('our_mission', 'Our Mission', function ($json) use($pageJson){
                        $json->textarea('heading', 'Heading');
                        $json->image('image', 'Side Image')
                                ->uniqueName()
                                ->move('sections')
                                ->help('Upload image for the section')
                                ->value($pageJson['our_mission']['image'] ?? null);
                            if(isset($pageJson['our_mission']['image'])) {
                                $json->hidden('current_image', 'Current Image')
                                    ->default($pageJson['our_mission']['image'])
                                    ->readonly();
                            }
                        for($i=0; $i<3; $i++){
                            $json->embeds('list_'.$i, 'Icon List '.$i+1, function ($json) use($pageJson){
                                $json->text('icon', 'Fontawesome Icon class');
                                $json->text('title', 'Title');
                                $json->text('paragraph', 'Paragraph');
                            });
                        }
                    });

                    //Our team ------------------------------------------------------------------------
                    $json->divider('Our Team');
                    for($i=0; $i<5; $i++){
                        $json->embeds('team_member_'.$i, 'Team Member '.$i+1, function ($json) use($pageJson, $i){
                            $json->text('name', 'Name');
                            $json->text('designation', 'Designation');
                            $json->textarea('details', 'Details');
                            $json->url('instagram');
                            $json->url('linked_in');
                            $json->url('whatsapp');
                            $json->image('image', 'Profile Image')
                                    ->uniqueName()
                                    ->move('sections')
                                    ->help('Upload image for the section')
                                    ->value($pageJson['team_member_'.$i]['image'] ?? null);
                                if(isset($pageJson['team_member_'.$i]['image'])) {
                                    $json->hidden('current_image', 'Current Image')
                                        ->default($pageJson['team_member_'.$i]['image'])
                                        ->readonly();
                                }
                        });
                    }

                    // our partners ------------------------------------------------------------------------
                    $json->divider('Partners Image');
                    $currentPartnersImages = $pageJson['partners_image'] ?? [];
                    $json->multipleImage('partners_image', 'Partners Image')
                        ->uniqueName()
                        ->move('sections')
                        ->help('Upload images for the partners section')
                        ->value($currentPartnersImages); // Pass array directly
                    $json->hidden('partners_current_image', 'Current Images')
                        ->default(json_encode($currentPartnersImages)) // Encode the array
                        ->readonly();


                });
            }
            // contact-us page --------------------------------------------------------------------------
            if ($form->model()->slug == 'contact-us') {
                $form->embeds('page_json', 'Contact Us', function ($json) use($pageJson) {
                    $json->text('heading', 'Section Heading')->required();
                    $json->text('sub_heading', 'Sub Heading')->required();
                    $json->image('image', 'Side Image')
                        ->uniqueName()
                        ->move('sections')
                        ->help('Upload image for the section')
                        ->value($pageJson['image'] ?? null);
                    if(isset($pageJson['image'])) {
                        $json->hidden('current__image', 'Current Image')
                            ->default($pageJson['image'])
                            ->readonly();
                    }
                    $json->embeds('form', 'Form Content', function ($json) {
                        $json->text('heading', 'Section Heading')->required();
                        $json->text('sub_heading', 'Sub Heading')->required();
                        $json->text('button_text', 'Button Text')->required();
                    });
                });
            }
        });

        $form->saving(function (Form $form) {
            if (request('page_json')) {
                $form->model()->page_json = request('page_json');
            }
        });

        return $form;
    }

}
