<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Page extends Model
{
    protected $guarded = [];

    protected $casts = [
        'page_json' => 'array',
    ];

    public function sections()
    {
        return $this->hasMany(PageSection::class);
    }

    protected static function boot()
    {
        parent::boot();

        static::saving(function ($page) {
            if (isset($page->name)) {
                $page->slug = \Str::slug($page->name);
            }
        });
    }

    public function setPageJsonAttribute($value){
        // Ensure we have an array
        $pageJson = is_array($value) ? $value : json_decode($value, true) ?? [];

        // about-us page -----------------------------------------------------------------------
        if($this->slug === 'about-us'){
            
            // Handle founder image
            if (isset($pageJson['founder_image']) && $pageJson['founder_image'] instanceof \Illuminate\Http\UploadedFile) {
                $imagePath = $pageJson['founder_image']->store('sections', 'public');
                $pageJson['founder_image'] = $imagePath;
            } elseif (isset($pageJson['current_founder_image']) && is_string($pageJson['current_founder_image'])) {
                $pageJson['founder_image'] = $pageJson['current_founder_image'];
            } else {
                $pageJson['founder_image'] = null;
            }

            // Handle partners multiple images
            if (isset($pageJson['partners_image'])) {
                $partnersImages = [];
                
                // Process each uploaded file
                if (is_array($pageJson['partners_image'])) {
                    foreach ($pageJson['partners_image'] as $image) {
                        if ($image instanceof \Illuminate\Http\UploadedFile) {
                            $imagePath = $image->store('sections', 'public');
                            $partnersImages[] = $imagePath;
                        } elseif (is_string($image)) {
                            // Retain existing image paths
                            $partnersImages[] = $image;
                        }
                    }
                }
                
                // If no new images were uploaded but we have current images, use them
                if (empty($partnersImages) && isset($pageJson['partners_current_image'])) {
                    $currentImages = json_decode($pageJson['partners_current_image'], true) ?? [];
                    if (is_array($currentImages)) {
                        $partnersImages = $currentImages;
                    }
                }
                
                $pageJson['partners_image'] = $partnersImages;
            } else {
                $pageJson['partners_image'] = [];
            }

            // Handle image-box cards (4 cards)
            for ($i = 0; $i < 4; $i++) {
                $cardKey = 'card_' . $i;
                if (isset($pageJson[$cardKey]['image']) && $pageJson[$cardKey]['image'] instanceof \Illuminate\Http\UploadedFile) {
                    $imagePath = $pageJson[$cardKey]['image']->store('sections', 'public');
                    $pageJson[$cardKey]['image'] = $imagePath;
                } elseif (isset($pageJson[$cardKey]['current_image']) && is_string($pageJson[$cardKey]['current_image'])) {
                    $pageJson[$cardKey]['image'] = $pageJson[$cardKey]['current_image'];
                } else {
                    $pageJson[$cardKey]['image'] = null;
                }

                // Ensure other card fields exist
                $pageJson[$cardKey]['title'] = $pageJson[$cardKey]['title'] ?? null;
                $pageJson[$cardKey]['paragraph'] = $pageJson[$cardKey]['paragraph'] ?? null;
            }

            // Handle image-title sections (3 sections)
            for ($i = 0; $i < 3; $i++) {
                $sectionKey = 'section_' . $i;
                if (isset($pageJson[$sectionKey]['image']) && $pageJson[$sectionKey]['image'] instanceof \Illuminate\Http\UploadedFile) {
                    $imagePath = $pageJson[$sectionKey]['image']->store('sections', 'public');
                    $pageJson[$sectionKey]['image'] = $imagePath;
                } elseif (isset($pageJson[$sectionKey]['current_image']) && is_string($pageJson[$sectionKey]['current_image'])) {
                    $pageJson[$sectionKey]['image'] = $pageJson[$sectionKey]['current_image'];
                } else {
                    $pageJson[$sectionKey]['image'] = null;
                }

                // Ensure title field exists
                $pageJson[$sectionKey]['title'] = $pageJson[$sectionKey]['title'] ?? null;
            }

            // Handle our mission section
            if (isset($pageJson['our_mission']['image']) && $pageJson['our_mission']['image'] instanceof \Illuminate\Http\UploadedFile) {
                $imagePath = $pageJson['our_mission']['image']->store('sections', 'public');
                $pageJson['our_mission']['image'] = $imagePath;
            } elseif (isset($pageJson['our_mission']['current_image']) && is_string($pageJson['our_mission']['current_image'])) {
                $pageJson['our_mission']['image'] = $pageJson['our_mission']['current_image'];
            } else {
                $pageJson['our_mission']['image'] = null;
            }

            // Handle our mission lists (3 lists)
            for ($i = 0; $i < 3; $i++) {
                $listKey = 'list_' . $i;
                if (isset($pageJson['our_mission'][$listKey])) {
                    $pageJson['our_mission'][$listKey]['icon'] = $pageJson['our_mission'][$listKey]['icon'] ?? null;
                    $pageJson['our_mission'][$listKey]['title'] = $pageJson['our_mission'][$listKey]['title'] ?? null;
                    $pageJson['our_mission'][$listKey]['paragraph'] = $pageJson['our_mission'][$listKey]['paragraph'] ?? null;
                }
            }

            // Handle team members (5 members)
            for ($i = 0; $i < 5; $i++) {
                $memberKey = 'team_member_' . $i;
                if (isset($pageJson[$memberKey]['image']) && $pageJson[$memberKey]['image'] instanceof \Illuminate\Http\UploadedFile) {
                    $imagePath = $pageJson[$memberKey]['image']->store('sections', 'public');
                    $pageJson[$memberKey]['image'] = $imagePath;
                } elseif (isset($pageJson[$memberKey]['current_image']) && is_string($pageJson[$memberKey]['current_image'])) {
                    $pageJson[$memberKey]['image'] = $pageJson[$memberKey]['current_image'];
                } else {
                    $pageJson[$memberKey]['image'] = null;
                }

                // Ensure other team member fields exist
                $pageJson[$memberKey]['name'] = $pageJson[$memberKey]['name'] ?? null;
                $pageJson[$memberKey]['designation'] = $pageJson[$memberKey]['designation'] ?? null;
                $pageJson[$memberKey]['instagram'] = $pageJson[$memberKey]['instagram'] ?? null;
                $pageJson[$memberKey]['linked_in'] = $pageJson[$memberKey]['linked_in'] ?? null;
                $pageJson[$memberKey]['whatsapp'] = $pageJson[$memberKey]['whatsapp'] ?? null;
            }

            // Ensure all required fields exist with default values
            $pageJson['sub_heading'] = $pageJson['sub_heading'] ?? null;
            $pageJson['heading'] = $pageJson['heading'] ?? null;
            $pageJson['paragraph'] = $pageJson['paragraph'] ?? null;
            $pageJson['year_count'] = $pageJson['year_count'] ?? null;
            $pageJson['customer_served'] = $pageJson['customer_served'] ?? null;
            $pageJson['successfull_starts'] = $pageJson['successfull_starts'] ?? null;
            $pageJson['satisfaction_rate'] = $pageJson['satisfaction_rate'] ?? null;
            $pageJson['founder_details'] = $pageJson['founder_details'] ?? null;

        }

        // contact-us page --------------------------------------------------------------------
        if ($this->slug == 'contact-us') {
            // Handle founder image
            if (isset($pageJson['image']) && $pageJson['image'] instanceof \Illuminate\Http\UploadedFile) {
                $imagePath = $pageJson['image']->store('sections', 'public');
                $pageJson['image'] = $imagePath;
            } elseif (isset($pageJson['current_image']) && is_string($pageJson['current_image'])) {
                $pageJson['image'] = $pageJson['current_image'];
            } else {
                $pageJson['image'] = null;
            }
            
            if (isset($pageJson['form']) && is_array($pageJson['form'])) {
                $pageJson['form'] = array_filter($pageJson['form']);
            }
        }

        $this->attributes['page_json'] = json_encode($pageJson);
    }


    public function getPageJsonAttribute($value)
    {
        $pageJson = json_decode($value, true) ?? [];
        
        // Ensure all expected keys exist to prevent undefined index errors
        $defaultStructure = [
            'sub_heading' => null,
            'heading' => null,
            'paragraph' => null,
            'year_count' => null,
            'founder_image' => null,
            'founder_details' => null,
            'customer_served' => null,
            'successfull_starts' => null,
            'satisfaction_rate' => null,
            'our_mission' => [
                'heading' => null,
                'image' => null,
                'list_0' => ['icon' => null, 'title' => null, 'paragraph' => null],
                'list_1' => ['icon' => null, 'title' => null, 'paragraph' => null],
                'list_2' => ['icon' => null, 'title' => null, 'paragraph' => null]
            ]
        ];

        // Add card structures
        for ($i = 0; $i < 4; $i++) {
            $defaultStructure['card_' . $i] = [
                'image' => null,
                'title' => null,
                'paragraph' => null
            ];
        }

        // Add section structures
        for ($i = 0; $i < 3; $i++) {
            $defaultStructure['section_' . $i] = [
                'image' => null,
                'title' => null
            ];
        }

        // Add team member structures
        for ($i = 0; $i < 5; $i++) {
            $defaultStructure['team_member_' . $i] = [
                'name' => null,
                'designation' => null,
                'instagram' => null,
                'linked_in' => null,
                'whatsapp' => null,
                'image' => null
            ];
        }

        // Merge with actual data
        return array_merge($defaultStructure, $pageJson);
    }
    
}
